/*
 * VisuProxy  : part of P2P-MPI (http://www.p2pmpi.org).
 *
 * Copyright (C) 2007, Stephane Genaud and Choopan Rattanapoka 
 *
 * This tool is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
   VisuProxy acts as a cache for SuperNode. Clients can register to VisuProxy
   and will be notified of changes on SuperNode through VisuProxy messages.


   SuperNode stores information about hosts running MPD.
   It is a simple replacement to the JXTA SRDI previously used.
   Information is a list of 'HostEntry':
   HostEntry(String ip, int mpdPort, int fdPort, int ftPort, int rsPort, int numProcPerJob)

   @author Stephane Genaud, creation jeudi 2 août 2007, 11:11:34 (UTC+0200)
 */


package p2pmpi.visu;

import p2pmpi.common.*;
import p2pmpi.message.*;
import p2pmpi.p2p.message.*;   // use RegisterMessage.class
import p2pmpi.tools.SupernodeHostCacheDisplay;
import java.util.*;
import java.net.*;
import java.io.*;

import org.apache.log4j.*;

public class VisuProxy extends Thread {

	  // options: always 4 arguments: short name,long name, parameter, explaination 
	  private static String options[]={
		    "-h","--help",    null, "Display this help message.",
		    "-u","--updateFreq", "n",  "Set update frequency to every n seconds."
	  };


	  private static final String portPropName = "VisuProxyListenPort"; // 
	  private static final String proxyConfFile = "P2P-RDV.conf"; // 
	  private static int listenPort;
	  private static int updateFreq        = 30;  // default value, in seconds
	  private URI supernodeUri             = null;
	  protected Vector<Client> clientList  = null;
	  protected Vector<HostEntry> peerList = null;
	  protected static Logger log;

	  // --------------------- C L I E N T --------------------------------------------------
	  private class Client {
		    public String id;
		    int answerPort;	// store port specified by client in its registration message
		    HostEntry hostEntry;
		    private long lastSeen;
		    private long firstSeen;

		    /**
		     * Constructor.
		     **/
		    public Client ( String id, HostEntry e, int answerPort ) {
				this.id = id;
				this.answerPort = answerPort;
				this.hostEntry = e;
				this.firstSeen = System.currentTimeMillis();
				this.lastSeen = this.firstSeen;
		    }

		    public String getIp() {
				return( hostEntry.getIp( ) );
		    }

		    public int getAnswerPort() {
				return( this.answerPort );
		    }

		    public String getId( ) {
				return ( this.id );
		    }
		    public boolean removeClient ( ) {
				Iterator iter = clientList.iterator();
				while ( iter.hasNext() ) {
					Object elem  = iter.next();
					String listElemId = ((Client) elem).getId(); 
					if (this.id.equals(listElemId)) {
						iter.remove();
						return( true );
					}
				}
				return( false );
		    }

		    /**
		     * Notify client of new peerList.
		     * @return the number of clients notified.
		     **/
		    public boolean notifyList ( ) {
				boolean done = false;
				if ( clientList != null ) {
					  // -- pack it in a serializable message
					  UpdateListMessage msg = new UpdateListMessage( peerList );
					  done =  sendUpdate( this.getIp(), this.getAnswerPort(), msg);
				}
				return( done );
		    }

	  }
	  // ------------------------ V I S U  P R O X Y ------------------------
	  /**
	   * Constructor 1.
	   **/
	  public VisuProxy () {
		    P2PMPI_ConfigFile ppConf = new P2PMPI_ConfigFile( );
		    this.supernodeUri = ppConf.getSuperNode();
		    this.clientList = new Vector<Client>();
		    log = Logger.getLogger("VISU");
	  }
	  /**
	   * Constructor 2. Modify default supernode queries frequency.
	   **/
	  public VisuProxy ( int refreqFreq ) {
		    this.updateFreq = Math.max( 5, refreqFreq ); // do not stress supernode too much ! 
		    P2PMPI_ConfigFile ppConf = new P2PMPI_ConfigFile( );
		    this.supernodeUri = ppConf.getSuperNode();
		    this.clientList = new Vector<Client>();
		    log = Logger.getLogger("VISU");
	  }

	  /**
	   * Proceed with the connection and sending to given host:port 
	   * @param host ip or hostname as a String
	   * @param port to send to on host
	   * @param o the object to send. Must be serializable.
	   * @return true if connection could be established, false otherwise 
	   **/
	  private boolean sendUpdate( String host, int port, Object o ) { 

		    OutputStream out = null;
		    ObjectOutputStream oos = null;
		    Socket s = null;
		    try {
				StatQueryMessage queryMsg = new StatQueryMessage();
				//System.out.println("@ send( "+host +","+ port+")");
				s = new Socket( );
				SocketAddress sa  = new InetSocketAddress( host , port);
				s.connect( sa, 5000 );  // 5s timeout
				out = s.getOutputStream();
				oos = new ObjectOutputStream(out);
				oos.writeObject( o );
				oos.flush();
				oos.close();
				out.close();
				s.close();
				return( true );
		    } catch (Exception e) { 
				//e.printStackTrace();
		    }
		    return( false );
	  }



	  /**
	   * Get a reference to a specific client Object in clientList given its id.
	   * @param id the String identifying the client object (e.g. hostname).
	   * @return the first object foudn with corresponding id, or null if none was found. 
	   **/
	  public Client getClient( String id ) {

		    for (int i=0; i<clientList.size(); i++) {
				if ( (clientList.get( i ).id).equals( id ) )
					  return( clientList.get( i ) );
		    }
		    return( null );

	  }


	  /**
	   *  VisuProxy client listening thread.
	   *  Listen and answer to incoming messages.
	   **/
	  public void run() {

		File proxyconf = new File(System.getProperty("P2PMPI_HOME"), proxyConfFile );
		GenericConfigFile conf = new GenericConfigFile( proxyconf.toString());
		// first guess on which port VisuProxy will listen
		String portConf = conf.getProperty( portPropName );
		if (portConf == null) { 
			  System.err.println("[Error] can not find a valid port under property \""+portPropName+"\" in configuration file "+proxyConfFile+". Exiting."); 
			  System.exit( 1 );
		}
		this.listenPort = Integer.parseInt( portConf );

		ServerSocket visuProxySocket = null;
		try {
			  String msg = "Listening on port " + listenPort;
			  System.out.println( msg );
			  log.info( "[VisuProxy] " + msg );
			  visuProxySocket = new ServerSocket( listenPort );
		} catch (Exception e) {
			  e.printStackTrace();
			  return;
		}


		// ---- create a thread to dialog with Supernode  -----
		File configFile = new File( System.getProperty("P2PMPI_HOME"), "P2P-MPI.conf" );
		P2PMPI_ConfigFile ppConf = new P2PMPI_ConfigFile( configFile.toString() );
		URI supernodeUri     = ppConf.getSuperNode( );
		System.out.println("Visu Proxy starting connection thread with supernode...");
		ContactSupernodeThread snthread = new ContactSupernodeThread( supernodeUri );
		snthread.setName("Connection with Supernode thread");
		snthread.start();


		// ---- permanently listen to registration/unrgistration requests -----
		while(true) {
			  try {
				    Socket conn = visuProxySocket.accept();
				    VisuProxyThread t = new VisuProxyThread( visuProxySocket, conn );
				    t.setName("VisuProxy Listening thread");
				    t.start();
			  } catch (Exception e) { e.printStackTrace(); break; }
			  // Exception raise when serverSocket is closed
			  System.gc();
		}
		try {
			  visuProxySocket.close();
		} catch (Exception e) {}
		System.exit( 1 ); 
	  }

	  /**
	   * usage.
	   **/
	  private static void usage() {
		    int i;
		    System.out.println("usage: runVisuProxy [OPTION...]");  
		    for(i=0;i<options.length;i+=4) {
				System.out.print(options[i]+"| "+options[i+1]+"\t");
				if (options[i+2]!=null) 
					  System.out.print(options[i+2]);
				System.out.println("\t"+options[i+3]);  
		    }
	  }

	  /**
	   * parse command line arguments.
	   *	 https://logitelnet.socgen.com/html/menu/indexlgn.html
	   * @param args the command line arguments
	   * @param optionNum the option (as described in options array) to detect in args
	   * @return null when option optionNum is not in the command line arguments 
	   *         If the option exists among arguments 
	   *         return either a value if option requires a parameter value
	   *         or an empty string if option does not require a value
	   **/
	  private static String parseArgs(String [] args,int optionNum) {
		    int i,
			  indx  = optionNum*4;  // indx in option array, based on 4 item / option

		    for (i=0;i<args.length;i++) {
				// match short or long option name ?
				if (args[i].equals(options[indx]) || args[i].equals(options[indx+1])) {
					  // option supposed to have a parameter value ?
					  if (options[indx+2] != null) {
						    if (args.length >= i+2 )
								return (args[i+1]); // Next arg is supposed to be the parameter
						    else {
								System.out.println("command line parse error: "+
										    options[indx]+" or "+ options[indx+1]+" requires a parameter.");
								System.exit(1);
						    }
					  }
					  else {
						    return (new String("")); // an empty string means option was selected,
						    // but no value.
					  }
				}
		    }
		    return (null);
	  }


	  //--------------------------------- main ------------------------------------------
	  public static void main ( String [] args ) {
		    VisuProxy s = null;

		    // command line args parsing
		    if ( args.length > 0 ) {
				String ret=null;
				// help option is in position 0
				if ( (ret=parseArgs(args,0)) != null ) {
					  usage();
					  System.exit(0);
				}

				// console mode option is in position 1
				String valUpdateFreq = parseArgs(args,1);
				if (valUpdateFreq != null) {
					  if(!valUpdateFreq.equals("")) {
						    System.out.println("Visu Proxy running (update every "+valUpdateFreq+"s)");
						    s = new VisuProxy( Integer.parseInt(valUpdateFreq ) );
					  }
				}
				else {
					  System.out.println("Bad command line arguments.");
					  usage();
					  System.exit(1);
				}
		    }
		    else { // no cmd line args
		    		System.out.println("Visu Proxy running ...");
				s = new VisuProxy( );
		    }
		    System.out.println("Visu Proxy starting listening thread ...");
		    s.start();
	  }


	  //-------------------- M E S S A G E   H A N D L I N G   T H R E A D ---------------------------


	  public class VisuProxyThread extends Thread {
		    ServerSocket serverSocket = null;
		    Socket socket = null;

		    VisuProxyThread( ServerSocket serverSocket, Socket socket ) {
				this.serverSocket = serverSocket;
				this.socket       = socket;
		    }

		    /**
		     * Thread handling connections.
		     * Wait for register/unregister messages from clients, and send notifications
		     * when changes occurs in SuperNode tables.
		     **/
		    public void run( ) {
				InputStream in = null;
				OutputStream out = null;
				ObjectInputStream ois = null;
				ObjectOutputStream oos = null;
				Object inMsg;
				HostEntry host;

				try {
					  in = socket.getInputStream();
					  out = socket.getOutputStream();
					  ois = new ObjectInputStream(in);
					  inMsg = ois.readObject();
				} catch (Exception e) {
					  return;
				}
				if ( inMsg instanceof RegisterMessage ) {
					  host = ((RegisterMessage)inMsg).getHostEntry();
					  String ip = host.getIp();
					  int answerPort = ((RegisterMessage)inMsg).getAnswerPort();
					  log.info("[VisuProxy] register from " + ip + "(answer on port "+ answerPort+")" );
					  Client client = new Client( ip, host, answerPort );
					  synchronized ( clientList ) {
						    clientList.add( client );
					  }
					  // send a special update just afer registration (for impatient users !)
					  client.notifyList( );
				}
				else if (inMsg instanceof UnregisterMessage) {
					  host = ((UnregisterMessage)inMsg).getHostEntry();
					  log.info("[VisuProxy] unregister from " + host.getIp());
					  Client client = getClient( host.getIp() );
					  if ( client == null ) {
						    log.error("[VisuProxy] cannot find client with id="+ host.getIp());
					  }
					  else {
						    Boolean res;
						    synchronized ( clientList ) {
								res = client.removeClient( );
						    }
						    if (res)
								log.debug("[VisuProxy] removing " + host.getIp() + " from clientList (#records="+clientList.size());
						    else 
								log.debug("[VisuProxy] could not remove client.");
					  }
				}
				else if ( inMsg instanceof QuitMessage ) {
					  log.info("[VisuProxy] instructed to quit. Message from "+((QuitMessage)inMsg).getSrc( ));
/*
					  try {
						    this.serverSocket.close();
					  }
					  catch (Exception e) { }
*/
					  System.exit( 0 );
				}
		    }

	  }

	  //-------------------- C O M M U N I C A T I O N   S U P E R N O D E   T H R E A D ---------------------------

	  public class ContactSupernodeThread extends Thread {
		    URI supernodeUri = null;

		    public ContactSupernodeThread( URI supernodeUri ) {
				this.supernodeUri =  supernodeUri;
		    }

		    /**
		     * Notify registered clients of new peerList.
		     * @return the number of clients notified.
		     **/
		    public int notifyListToClients( ) {
				int numNotifiedClients = 0;
				if ( clientList == null )
					return( 0 );
				Iterator iter = clientList.iterator();
				while ( iter.hasNext() ) {
					  Object elem  = iter.next();
					  if ( ((Client) elem).notifyList( ) ) {
						numNotifiedClients++;
					  }
				}
				return( numNotifiedClients );
		    }


		    /**
		     * Thread handling communication with supernode. 
		     **/
		    public void run( ) {

				Vector<HostEntry> freshPeerList = null;				

				// --- permanently queries Supernode about connected peers 
				while( true ) {
					
					  if ( (freshPeerList = SupernodeHostCacheDisplay.query( supernodeUri.getHost(), supernodeUri.getPort() )) != null ) {
						    //SupernodeHostCacheDisplay.display( freshPeerList );
					  }
					  else {
						    log.error("[VisuProxy] Could not query SuperNode at "+  supernodeUri.toString() + ". Exiting.");
						    System.exit(1);
					  }
					  // --- check if the set of connected peers has changed.
					  if ( (peerList == null) || (freshPeerList.hashCode() != peerList.hashCode()) ) {
							peerList = freshPeerList;
							notifyListToClients();
					  }
					  // --- sleep for a while. Give Supernode a rest !
					  try {
						    Thread.sleep( updateFreq*1000 );
					  }
					  catch (Exception e) {
						    e.printStackTrace( );
					  }
				}
		    }
	  }
}
