/*
 * SuperNodeHostCacheDisplay  : part of P2P-MPI (http://www.p2pmpi.org).
 *
 * Copyright (C) 2007, Choopan Rattanapoka 
 *
 * This tool is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 SuperNodeHostCacheDisplay allows to get or display the contents of a SuperNode's tables.
 
 
 SuperNode stores information about hosts running MPD.
 It is a simple replacement to the JXTA SRDI previously used.
 Information is a list of 'HostEntry':
 HostEntry(String ip, int mpdPort, int fdPort, int ftPort, int rsPort, int numProcPerJob)

 @author Choopan Rattanapoka, creation july 2007
 @author Stephane Genaud, modification jul 23 10:09:59 CEST 2007
*/


package p2pmpi.tools;

import p2pmpi.common.*;
import p2pmpi.message.*;
import p2pmpi.p2p.message.*;
import java.util.*;
import java.net.*;
import java.io.*;

public class SupernodeHostCacheDisplay {
	static private String dateTranslate(long time) {
		long tmp = time;

		int day = (int)(time/(84600*1000));
		tmp -= (day * (84600*1000));
		int hour = (int)(tmp/(3600*1000));
		tmp -= (hour * (3600*1000));
		int min = (int)(tmp/(60*1000));
		tmp -= (min * (60*1000));
		int sec = (int)(tmp/1000);
		tmp -= (sec * 1000);
		int ms = (int)tmp;

		return day+"D " + hour + ":" + min + ":" + sec + ":" + ms;
	}

	static public void display(Vector<HostEntry> hostCache) {
		System.out.println("Host\t\tMPD Port\tLast update");
		if(hostCache == null) {
			return;
		}
		int size = hostCache.size();
		for(int i = 0; i < size; i++) {
			HostEntry host = hostCache.elementAt(i);
			System.out.println(host.getIp() + "\t" + host.getMpdPort() + "\t\t" + dateTranslate(host.getLastUpdate()));
		}
		System.out.println("\n\nTotal : " + size + " MPD known.");
	}

	
	/**
	 * Query a remote SuperNode and get a list of known peers.
	 * 
	 * @param host the IP or name of host running the SperNode
	 * @param port the host running the SperNode is listening to
	 * @return a vector of HostEntry containing the information, or null if SuperNode could not be contacted.
	 **/
	static public  Vector<HostEntry> query( String host , int port ) {

		Vector<HostEntry> hostCache = null;

		try {
			PeerRequestMessage requestMsg = new PeerRequestMessage();
			Socket sock = new Socket( host , port );
			InputStream in   = sock.getInputStream();
			OutputStream out = sock.getOutputStream();
			ObjectOutputStream oos = new ObjectOutputStream(out);
			oos.writeObject(requestMsg);
			oos.flush();

			ObjectInputStream ois = new ObjectInputStream(in);
			PeerReplyMessage replyMsg = (PeerReplyMessage)ois.readObject();
			hostCache = replyMsg.getHostList();
			ois.close();
			oos.close();
			out.close();
			in.close();
			sock.close();
		} catch(Exception e) {
			System.err.println("[Error] querying Supernode : " + e.toString());
			return( null );
		}
		return( hostCache );

	}



	static public void main(String[] args) {
		String envPath = System.getProperty("P2PMPI_HOME");
		if(envPath == null) {
			System.err.println("Please make sure that the environment variable P2PMPI_HOME is set.");
			System.exit(1);
		}
		P2PMPI_ConfigFile ppConf = new P2PMPI_ConfigFile( );
		URI supernode 	= ppConf.getSuperNode();
		
		Vector<HostEntry> hostCache = null;
		if ( (hostCache = SupernodeHostCacheDisplay.query( supernode.getHost(), supernode.getPort() )) != null ) {
			SupernodeHostCacheDisplay.display( hostCache );
		}
		else {
			System.err.println("Could not query SuperNode at "+  supernode.getHost() + ":" + supernode.getPort());
			System.exit(1);
		}
	}
}
