/*
 * SuperNode : part of P2P-MPI (http://www.p2pmpi.org).
 *
 * Copyright (C) 2007, Choopan Rattanapoka and Stephane Genaud
 *
 * This tool is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 StatInfoMessage is used to vehicle information about the state of a MPD:
 the host it runs on, the host's OS, the number of applications currently running, ....
*/

package p2pmpi.message;

import p2pmpi.common.*;

import java.io.*;
import java.net.*;
import java.util.*;



public class StatInfoMessage implements Serializable {
	private static final long serialVersionUID = 2000033L;
	private Vector<ProcessInfo> procInfo;  // number of applications currently running
	String mpdVer;   // MPD version
	String ip;       // host IP
	int    mpdPort;  // mpdPort
	String hostname; // host name
	String osname;   // OS name
	String arch;     // architecture
	String cpuname;  // cpu name
      String cpufreq;  // cpu frequency;
      String mem;      // memory size;

	/**
	 * constructor 1. Set information variables about this peer
	 * using protected members of P2PMPI_MPDPeer plus System
	 * calls for physical or OS characteristics.
	 **/
	public StatInfoMessage( String ip, int mpdPort) {
		  this.ip = ip;
	        this.mpdPort = mpdPort;
		  this.mpdVer = Version.MAINVERSION+"."+Version.SUBVERSION;
		  this.procInfo = new Vector<ProcessInfo>();
		  this.osname   = System.getProperty("os.name");
 		  this.arch     = System.getProperty("os.arch");
	        String cpuinfo [] = OsInfo.cpuInfo();
              String meminfo [] = OsInfo.memInfo();
		  if ( meminfo != null ) {
			this.mem = meminfo[0];
		  }
              if (cpuinfo == null) {
                    this.cpuname = "N/A";
                    this.cpufreq = "N/A";
              }
              else {
                    this.cpuname = cpuinfo[0]; 
                    this.cpufreq = cpuinfo[1];
              }
	}
	/**
	 * constructor 2. Set static known information into object. 
	 * This constructor is mainly intended at tranfering information from peer cache.
	 **/
	public StatInfoMessage( String ip, int mpdPort, String version, String osName, String arch, String cpuName, String cpuFreq, String mem ) {
		  this.ip = ip;
	        this.mpdPort  = mpdPort;
		  this.mpdVer   = version;
		  this.procInfo = null; // can't guess running applications
		  this.osname   = osName; 
 		  this.arch     = arch; 
              this.cpuname  = cpuName;  
              this.cpufreq  = cpuFreq;
		  this.mem      = mem;
	}

	public void addInfo(ProcessInfo info) {
		  procInfo.addElement(info);
	}

	public ProcessInfo getInfo(int index) {
		  return procInfo.elementAt(index);
	}

      /**
       * Number of application processes running at the time of building the StatInfo message.
       **/
	public int size() {
              if ( procInfo != null )
		      return( procInfo.size());
              else 
                  return( 0 );
	}


	public String getIp()      { return( ip );  }
	public String getHostname(){ return( hostname );}
	public String getMpdVer()  { return( mpdVer );  }
	public int    getMpdPort() { return( mpdPort );  }
	public String getOsName()  { return( osname );  }
	public String getArch()    { return( arch );    };
	public String getCpuName() { return( cpuname ); };
	public String getCpuFreq() { return( cpufreq ); };
	public String getMem() 	   { return( mem );     };
}
